import express from 'express';
import { createServer } from 'http';
import { Server } from 'socket.io';
import mysql from 'mysql2/promise';
import cors from 'cors';
import path from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import WhatsAppService from './whatsapp-service.js';


const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Load .env file using absolute path
dotenv.config({ path: path.join(__dirname, '.env') });

const app = express();
const server = createServer(app);
const corsOrigins = process.env.CORS_ORIGIN ? process.env.CORS_ORIGIN.split(',').map(origin => origin.trim()) : ['http://localhost:5173', 'http://127.0.0.1:5173'];
const io = new Server(server, {
  cors: {
    origin: corsOrigins,
    methods: ['GET', 'POST'],
    credentials: true
  }
});
const PORT = process.env.PORT || 3000;
const JWT_SECRET = process.env.JWT_SECRET || 'your-secret-key-change-this-in-production';

// Middleware
app.use(cors({
  origin: corsOrigins,
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With']
}));
app.use(express.json());
app.use(express.static(path.join(__dirname, 'dist')));



// Database connection
const dbConfig = {
  host: process.env.DB_HOST,
  port: process.env.DB_PORT,
  user: process.env.DB_USER,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME,
  ssl: process.env.DB_SSL === 'true' ? { rejectUnauthorized: false } : false,
};



// Database connection pool
let pool;
let whatsappService;



async function initializeDatabase() {
  try {
    pool = mysql.createPool({
      ...dbConfig,
      waitForConnections: true,
      connectionLimit: 10,
      queueLimit: 0,
    });

    // Create tables if they don't exist
    await pool.execute(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        mobile VARCHAR(15) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        api_key VARCHAR(255) UNIQUE NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_mobile (mobile),
        INDEX idx_api_key (api_key)
      )
    `);

    await pool.execute(`
      CREATE TABLE IF NOT EXISTS connected_devices (
        id INT AUTO_INCREMENT PRIMARY KEY,
        device_id VARCHAR(255) UNIQUE NOT NULL,
        user_id INT NOT NULL,
        pushname VARCHAR(255),
        phone_number VARCHAR(20),
        profile_picture_url TEXT,
        platform VARCHAR(50),
        is_connected BOOLEAN DEFAULT FALSE,
        connected_at TIMESTAMP NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        INDEX idx_user_id (user_id),
        INDEX idx_device_id (device_id),
        INDEX idx_is_connected (is_connected)
      )
    `);





    console.log('Database initialized successfully');
  } catch (error) {
    console.error('Database initialization error:', error);
  }
}

// Initialize WhatsApp Service
// Initialize WhatsApp Service
async function initializeWhatsAppService() {
  whatsappService = new WhatsAppService(io);

  try {
    // Fetch all devices that were marked as connected
    const [rows] = await pool.execute(
      'SELECT device_id, user_id FROM connected_devices WHERE is_connected = TRUE'
    );

    console.log(`Restoring ${rows.length} WhatsApp sessions...`);

    for (const row of rows) {
      whatsappService.connect(row.device_id, row.user_id).catch(err => {
        console.error(`Failed to restore session ${row.device_id}:`, err.message);
      });
    }
  } catch (error) {
    console.error('Error restoring WhatsApp sessions:', error);
  }
}

// Authentication middleware - supports both JWT tokens and API keys
const authenticateToken = async (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) {
    return res.status(401).json({ error: 'Access token required' });
  }

  try {
    // First, try to validate as JWT token
    try {
      const decoded = jwt.verify(token, JWT_SECRET);

      // Verify user exists in database
      const [rows] = await pool.execute(
        'SELECT id, mobile, api_key FROM users WHERE id = ?',
        [decoded.userId]
      );

      if (rows.length === 0) {
        return res.status(401).json({ error: 'Invalid token' });
      }

      req.user = rows[0];
      return next();
    } catch (jwtError) {
      // If JWT validation fails, try as API key
      const [rows] = await pool.execute(
        'SELECT id, mobile, api_key FROM users WHERE api_key = ?',
        [token]
      );

      if (rows.length === 0) {
        return res.status(401).json({ error: 'Invalid API key' });
      }

      req.user = rows[0];
      return next();
    }
  } catch (error) {
    console.error('Authentication error:', error);
    return res.status(500).json({ error: 'Authentication failed' });
  }
};

// Authentication API endpoints
app.post('/api/auth/register', async (req, res) => {
  try {
    const { mobile, password } = req.body;

    if (!mobile || !password) {
      return res.status(400).json({ error: 'Mobile and password are required' });
    }

    // Check if user already exists
    const [existingUsers] = await pool.execute(
      'SELECT id FROM users WHERE mobile = ?',
      [mobile]
    );

    if (existingUsers.length > 0) {
      return res.status(409).json({ error: 'Mobile number already registered' });
    }

    // Hash password
    const bcryptRounds = parseInt(process.env.BCRYPT_ROUNDS) || 10;
    const hashedPassword = await bcrypt.hash(password, bcryptRounds);

    // Generate API key
    const apiKey = jwt.sign({ mobile }, JWT_SECRET, { expiresIn: '365d' });

    // Insert new user
    const [result] = await pool.execute(
      'INSERT INTO users (mobile, password, api_key) VALUES (?, ?, ?)',
      [mobile, hashedPassword, apiKey]
    );

    res.status(201).json({
      message: 'User registered successfully',
      user: {
        id: result.insertId,
        mobile,
        api_key: apiKey
      }
    });
  } catch (error) {
    console.error('Registration error:', error);
    res.status(500).json({ error: 'Failed to register user' });
  }
});

app.post('/api/auth/login', async (req, res) => {
  try {
    const { mobile, password } = req.body;

    if (!mobile || !password) {
      return res.status(400).json({ error: 'Mobile and password are required' });
    }

    // Find user by mobile
    const [users] = await pool.execute(
      'SELECT id, mobile, password, api_key FROM users WHERE mobile = ?',
      [mobile]
    );

    if (users.length === 0) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    const user = users[0];

    // Verify password
    const isValidPassword = await bcrypt.compare(password, user.password);
    if (!isValidPassword) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    // Generate JWT token
    const token = jwt.sign({ userId: user.id, mobile: user.mobile }, JWT_SECRET, { expiresIn: '7d' });

    res.json({
      message: 'Login successful',
      token,
      user: {
        id: user.id,
        mobile: user.mobile,
        api_key: user.api_key
      }
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Failed to login' });
  }
});

// Protected routes
app.get('/api/auth/profile', authenticateToken, async (req, res) => {
  try {
    const [users] = await pool.execute(
      'SELECT id, mobile, api_key, created_at FROM users WHERE id = ?',
      [req.user.id]
    );

    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json({ user: users[0] });
  } catch (error) {
    console.error('Profile error:', error);
    res.status(500).json({ error: 'Failed to fetch profile' });
  }
});

// Regenerate API key
app.post('/api/auth/regenerate-api-key', authenticateToken, async (req, res) => {
  try {
    const newApiKey = jwt.sign({ mobile: req.user.mobile }, JWT_SECRET, { expiresIn: '365d' });

    await pool.execute(
      'UPDATE users SET api_key = ? WHERE id = ?',
      [newApiKey, req.user.id]
    );

    res.json({
      message: 'API key regenerated successfully',
      api_key: newApiKey
    });
  } catch (error) {
    console.error('Regenerate API key error:', error);
    res.status(500).json({ error: 'Failed to regenerate API key' });
  }
});

// Change mobile number
app.put('/api/auth/change-mobile', authenticateToken, async (req, res) => {
  try {
    const { newMobile, password } = req.body;

    if (!newMobile || !password) {
      return res.status(400).json({ error: 'New mobile number and password are required' });
    }

    // Verify current password
    const [users] = await pool.execute(
      'SELECT password FROM users WHERE id = ?',
      [req.user.id]
    );

    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const isValidPassword = await bcrypt.compare(password, users[0].password);
    if (!isValidPassword) {
      return res.status(401).json({ error: 'Invalid password' });
    }

    // Check if new mobile number already exists
    const [existingUsers] = await pool.execute(
      'SELECT id FROM users WHERE mobile = ? AND id != ?',
      [newMobile, req.user.id]
    );

    if (existingUsers.length > 0) {
      return res.status(409).json({ error: 'Mobile number already in use' });
    }

    // Update mobile number
    await pool.execute(
      'UPDATE users SET mobile = ? WHERE id = ?',
      [newMobile, req.user.id]
    );

    res.json({
      message: 'Mobile number updated successfully',
      mobile: newMobile
    });
  } catch (error) {
    console.error('Change mobile error:', error);
    res.status(500).json({ error: 'Failed to change mobile number' });
  }
});

// Change password
app.put('/api/auth/change-password', authenticateToken, async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;

    if (!currentPassword || !newPassword) {
      return res.status(400).json({ error: 'Current password and new password are required' });
    }

    if (newPassword.length < 6) {
      return res.status(400).json({ error: 'New password must be at least 6 characters' });
    }

    // Verify current password
    const [users] = await pool.execute(
      'SELECT password FROM users WHERE id = ?',
      [req.user.id]
    );

    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }

    const isValidPassword = await bcrypt.compare(currentPassword, users[0].password);
    if (!isValidPassword) {
      return res.status(401).json({ error: 'Current password is incorrect' });
    }

    // Hash new password
    const bcryptRounds = parseInt(process.env.BCRYPT_ROUNDS) || 10;
    const hashedNewPassword = await bcrypt.hash(newPassword, bcryptRounds);

    // Update password
    await pool.execute(
      'UPDATE users SET password = ? WHERE id = ?',
      [hashedNewPassword, req.user.id]
    );

    res.json({
      message: 'Password changed successfully'
    });
  } catch (error) {
    console.error('Change password error:', error);
    res.status(500).json({ error: 'Failed to change password' });
  }
});






// WhatsApp API endpoints
app.get('/api/whatsapp/status', authenticateToken, (req, res) => {
  if (!whatsappService) {
    return res.status(503).json({ error: 'WhatsApp service not initialized' });
  }

  res.json(whatsappService.getConnectionStatus());
});

app.post('/api/whatsapp/connect', authenticateToken, async (req, res) => {
  if (!whatsappService) {
    return res.status(503).json({ error: 'WhatsApp service not initialized' });
  }

  try {
    await whatsappService.connect();
    res.json({ message: 'WhatsApp connection initiated' });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

app.post('/api/whatsapp/disconnect', authenticateToken, async (req, res) => {
  if (!whatsappService) {
    return res.status(503).json({ error: 'WhatsApp service not initialized' });
  }

  try {
    await whatsappService.disconnect();
    res.json({ message: 'WhatsApp disconnected successfully' });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Message sending endpoints
app.post('/api/messages/send', authenticateToken, async (req, res) => {
  try {
    const { to, message, deviceId } = req.body;

    if (!to || !message) {
      return res.status(400).json({
        success: false,
        error: 'Recipient (to) and message are required'
      });
    }

    if (!whatsappService) {
      return res.status(503).json({
        success: false,
        error: 'WhatsApp service not initialized'
      });
    }

    // Check if device belongs to user and is connected
    const [devices] = await pool.execute(
      'SELECT device_id, is_connected FROM connected_devices WHERE device_id = ? AND user_id = ?',
      [deviceId || 'default', req.user.id]
    );

    if (devices.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Device not found or does not belong to user'
      });
    }

    if (!devices[0].is_connected) {
      return res.status(400).json({
        success: false,
        error: 'Device is not connected to WhatsApp'
      });
    }

    // Send message using WhatsApp service
    const result = await whatsappService.sendMessage(to, message, deviceId);

    res.json({
      success: true,
      message: 'Message sent successfully',
      data: result
    });

  } catch (error) {
    console.error('Error sending message:', error);
    res.status(500).json({
      success: false,
      error: error.message || 'Failed to send message'
    });
  }
});

// Send bulk messages
app.post('/api/messages/send-bulk', authenticateToken, async (req, res) => {
  try {
    const { messages, deviceId } = req.body;

    if (!Array.isArray(messages) || messages.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'Messages array is required and cannot be empty'
      });
    }

    if (!whatsappService) {
      return res.status(503).json({
        success: false,
        error: 'WhatsApp service not initialized'
      });
    }

    // Validate each message
    for (const msg of messages) {
      if (!msg.to || !msg.message) {
        return res.status(400).json({
          success: false,
          error: 'Each message must have "to" and "message" fields'
        });
      }
    }

    // Check if device belongs to user and is connected
    const [devices] = await pool.execute(
      'SELECT device_id, is_connected FROM connected_devices WHERE device_id = ? AND user_id = ?',
      [deviceId || 'default', req.user.id]
    );

    if (devices.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Device not found or does not belong to user'
      });
    }

    if (!devices[0].is_connected) {
      return res.status(400).json({
        success: false,
        error: 'Device is not connected to WhatsApp'
      });
    }

    // Send bulk messages
    const results = await Promise.allSettled(
      messages.map(msg => whatsappService.sendMessage(msg.to, msg.message, deviceId))
    );

    const successful = results.filter(r => r.status === 'fulfilled').map(r => r.value);
    const failed = results.filter(r => r.status === 'rejected').map(r => ({ error: r.reason.message }));

    res.json({
      success: true,
      message: 'Bulk messages processed',
      data: {
        successful: successful.length,
        failed: failed.length,
        results: {
          successful,
          failed
        }
      }
    });

  } catch (error) {
    console.error('Error sending bulk messages:', error);
    res.status(500).json({
      success: false,
      error: error.message || 'Failed to send bulk messages'
    });
  }
});

// Get message history (placeholder - would need message storage)
app.get('/api/messages/history', authenticateToken, async (req, res) => {
  try {
    const { deviceId, limit = 50, offset = 0 } = req.query;

    // For now, return empty array as we don't store message history
    // In a real implementation, you'd query a messages table
    res.json({
      success: true,
      messages: [],
      pagination: {
        limit: parseInt(limit),
        offset: parseInt(offset),
        total: 0
      }
    });

  } catch (error) {
    console.error('Error fetching message history:', error);
    res.status(500).json({
      success: false,
      error: error.message || 'Failed to fetch message history'
    });
  }
});



// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({
    success: true,
    status: 'healthy',
    timestamp: new Date().toISOString(),
    version: '1.0.0'
  });
});

// Device management endpoints
app.get('/api/devices', authenticateToken, async (req, res) => {
  try {
    const [rows] = await pool.execute(
      `SELECT device_id, pushname, phone_number, profile_picture_url, 
              platform, is_connected, connected_at, created_at
       FROM connected_devices 
       WHERE user_id = ? 
       ORDER BY connected_at DESC`,
      [req.user.id]
    );
    res.json({ success: true, devices: rows });
  } catch (error) {
    console.error('Error fetching devices:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

app.delete('/api/devices/:deviceId', authenticateToken, async (req, res) => {
  try {
    const { deviceId } = req.params;

    // Delete device from database
    await pool.execute(
      'DELETE FROM connected_devices WHERE device_id = ? AND user_id = ?',
      [deviceId, req.user.id]
    );

    // Delete auth files for this device
    const authPath = `./baileys_auth_info_${deviceId}`;
    const fs = require('fs');
    const path = require('path');

    if (fs.existsSync(authPath)) {
      fs.rmSync(authPath, { recursive: true, force: true });
    }

    res.json({ success: true, message: 'Device removed successfully' });
  } catch (error) {
    console.error('Error removing device:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

app.post('/api/whatsapp/connect/:deviceId', authenticateToken, async (req, res) => {
  try {
    const { deviceId } = req.params;
    await whatsappService.connect(deviceId, req.user.id);
    res.json({ success: true, message: 'WhatsApp connection initiated' });
  } catch (error) {
    console.error('Error connecting WhatsApp:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Serve React app for any other routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'dist', 'index.html'));
});

// Handle 404 for API routes (after all other routes)
app.use('/api/*', (req, res) => {
  res.status(404).json({ error: 'API endpoint not found' });
});

// Global JSON error handler - ensure all errors return JSON
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).json({ error: err.message || 'Internal server error' });
});

// Socket.IO connection handling
io.on('connection', (socket) => {
  console.log('Client connected:', socket.id);

  socket.on('join-device', (deviceId) => {
    socket.join(`device-${deviceId}`);
    console.log(`Client ${socket.id} joined device room: device-${deviceId}`);
  });

  socket.on('leave-device', (deviceId) => {
    socket.leave(`device-${deviceId}`);
    console.log(`Client ${socket.id} left device room: device-${deviceId}`);
  });

  socket.on('disconnect', () => {
    console.log('Client disconnected:', socket.id);
  });
});

// Request logging middleware
app.use((req, res, next) => {
  console.log(`${new Date().toISOString()} - ${req.method} ${req.path} - ${req.ip}`);
  next();
});

// Initialize and start server
async function startServer() {
  try {
    await initializeDatabase();
    console.log('✅ Database initialized successfully');

    await initializeWhatsAppService();
    console.log('✅ WhatsApp service initialized');

    server.listen(PORT, () => {
      console.log(`🚀 Server running on port ${PORT}`);
      console.log(`📍 Environment: ${process.env.NODE_ENV || 'development'}`);
      console.log(`🔗 CORS Origins: ${corsOrigins.join(', ')}`);
      console.log(`📊 Available endpoints:`);
      console.log(`   - GET  /api/health`);
      console.log(`   - POST /api/auth/login`);
      console.log(`   - GET  /api/devices`);
      console.log(`   - GET  /api/whatsapp/status`);
      console.log(`   - Socket.IO ready on port ${PORT}`);
    });
  } catch (error) {
    console.error('❌ Failed to start server:', error);
    process.exit(1);
  }
}

startServer().catch(console.error);

export default app;