import { makeWASocket, useMultiFileAuthState, DisconnectReason, makeCacheableSignalKeyStore, fetchLatestBaileysVersion } from '@whiskeysockets/baileys';
import { Boom } from '@hapi/boom';
import NodeCache from 'node-cache';
import P from 'pino';
import mysql from 'mysql2/promise';
import dotenv from 'dotenv';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

dotenv.config();

const logger = P({ level: 'error' });
const msgRetryCounterCache = new NodeCache();

const dbConfig = {
  host: process.env.DB_HOST,
  port: process.env.DB_PORT,
  user: process.env.DB_USER,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME,
  ssl: process.env.DB_SSL === 'true' ? { rejectUnauthorized: false } : false,
};

class WhatsAppService {
  constructor(io) {
    this.io = io;
    this.sockets = new Map(); // Store device_id -> socket
    this.connectionStatuses = new Map(); // Store device_id -> status
    this.qrs = new Map(); // Store device_id -> qr
    this.pool = null;
    this.initializeDatabase();
  }

  async initializeDatabase() {
    try {
      this.pool = mysql.createPool({
        ...dbConfig,
        waitForConnections: true,
        connectionLimit: 10,
        queueLimit: 0,
      });
      console.log('WhatsAppService database initialized');
    } catch (error) {
      console.error('WhatsAppService database error:', error);
    }
  }

  async connect(deviceId, userId = null) {
    if (!deviceId) return;

    try {
      const authPath = path.join(__dirname, `baileys_auth_info_${deviceId}`);
      const { state, saveCreds } = await useMultiFileAuthState(authPath);
      const { version } = await fetchLatestBaileysVersion();

      console.log(`Connecting device: ${deviceId} (WA v${version.join('.')})`);

      const sock = makeWASocket({
        version,
        logger,
        auth: {
          creds: state.creds,
          keys: makeCacheableSignalKeyStore(state.keys, logger),
        },
        msgRetryCounterCache,
        printQRInTerminal: false,
      });

      this.sockets.set(deviceId, sock);
      this.setupEventHandlers(sock, saveCreds, deviceId, userId);

      return sock;
    } catch (error) {
      console.error(`Error connecting device ${deviceId}:`, error);
      this.connectionStatuses.set(deviceId, 'error');
      this.io.emit('qr_error', { deviceId, error: error.message });
    }
  }

  setupEventHandlers(sock, saveCreds, deviceId, userId) {
    sock.ev.on('connection.update', async (update) => {
      const { connection, lastDisconnect, qr } = update;

      if (qr) {
        this.qrs.set(deviceId, qr);
        this.connectionStatuses.set(deviceId, 'qr_ready');
        this.io.emit('qr', { qr, deviceId });
      }

      if (connection === 'close') {
        const shouldReconnect = (lastDisconnect?.error instanceof Boom) ?
          lastDisconnect.error.output.statusCode !== DisconnectReason.loggedOut : true;

        this.connectionStatuses.set(deviceId, 'disconnected');
        await this.updateDeviceConnection(deviceId, false);

        if (shouldReconnect) {
          console.log(`Device ${deviceId} connection closed, reconnecting...`);
          setTimeout(() => this.connect(deviceId, userId), 5000);
        } else {
          console.log(`Device ${deviceId} logged out.`);
          this.sockets.delete(deviceId);
          this.connectionStatuses.set(deviceId, 'logged_out');
          this.io.emit('connection_status', { status: 'logged_out', deviceId });
        }
      } else if (connection === 'open') {
        console.log(`Device ${deviceId} connected successfully`);
        this.connectionStatuses.set(deviceId, 'connected');
        this.qrs.delete(deviceId);
        this.io.emit('connection_status', { status: 'connected', deviceId });

        if (userId) {
          await this.storeDeviceInfoWhenConnected(sock, deviceId, userId);
        }
      }
    });

    sock.ev.on('creds.update', saveCreds);

    sock.ev.on('messages.upsert', (m) => {
      this.io.emit('new_message', { deviceId, ...m });
    });
  }

  async storeDeviceInfoWhenConnected(sock, deviceId, userId) {
    try {
      const state = sock.authState.creds;
      const me = state.me;

      let profilePictureUrl = null;
      try {
        profilePictureUrl = await sock.profilePictureUrl(me.id);
      } catch (error) { }

      const actualPushname = me?.name || me?.pushName || me?.pushname || 'WhatsApp User';

      const deviceInfo = {
        pushname: actualPushname,
        phoneNumber: me.id.split('@')[0],
        profilePictureUrl: profilePictureUrl,
        platform: state.platform || 'web'
      };

      await this.storeDeviceInfo(userId, deviceId, deviceInfo);
      await this.updateDeviceConnection(deviceId, true);

      this.io.emit('device_connected', { deviceId, deviceInfo });
    } catch (error) {
      console.error('Error storing device info:', error);
    }
  }

  getConnectionStatus(deviceId) {
    return {
      status: this.connectionStatuses.get(deviceId) || 'disconnected',
      qr: this.qrs.get(deviceId) || null
    };
  }

  async disconnect(deviceId) {
    const sock = this.sockets.get(deviceId);
    if (sock) {
      try {
        await sock.logout();
      } catch (e) { }
      this.sockets.delete(deviceId);
    }
    this.connectionStatuses.set(deviceId, 'disconnected');
    this.qrs.delete(deviceId);
  }

  async storeDeviceInfo(userId, deviceId, deviceInfo) {
    if (!this.pool) return;
    try {
      await this.pool.execute(
        `INSERT INTO connected_devices 
         (device_id, user_id, pushname, phone_number, profile_picture_url, platform, is_connected, connected_at)
         VALUES (?, ?, ?, ?, ?, ?, TRUE, NOW())
         ON DUPLICATE KEY UPDATE 
         pushname = VALUES(pushname), phone_number = VALUES(phone_number), 
         profile_picture_url = VALUES(profile_picture_url), platform = VALUES(platform),
         is_connected = TRUE, connected_at = NOW()`,
        [deviceId, userId, deviceInfo.pushname, deviceInfo.phoneNumber, deviceInfo.profilePictureUrl, deviceInfo.platform]
      );
    } catch (error) {
      console.error('Error storing device info:', error);
    }
  }

  async updateDeviceConnection(deviceId, isConnected) {
    if (!this.pool) return;
    try {
      await this.pool.execute(
        'UPDATE connected_devices SET is_connected = ?, updated_at = NOW() WHERE device_id = ?',
        [isConnected, deviceId]
      );
    } catch (error) { }
  }

  async sendMessage(jid, message, deviceId) {
    const sock = this.sockets.get(deviceId);
    const status = this.connectionStatuses.get(deviceId);

    if (sock && status === 'connected') {
      const formattedJid = jid.includes('@') ? jid : `${jid.replace(/\D/g, '')}@s.whatsapp.net`;
      try {
        const result = await sock.sendMessage(formattedJid, { text: message });
        return {
          success: true,
          messageId: result.key.id,
          timestamp: result.messageTimestamp,
          to: formattedJid,
          deviceId: deviceId
        };
      } catch (error) {
        throw new Error(`Failed to send message: ${error.message}`);
      }
    }
    throw new Error('WhatsApp not connected for this device');
  }
}

export default WhatsAppService;